const Router = require("express").Router();
const { DB } = require("../../db");
const {
  ACCEPTED_CREATE_CARD_SCHEMA,
  ACCEPTED_EDIT_CARD_SCHEMA,
  ACCEPTED_CARD_TRANSFER,
} = require("../../validators/card");
const { verifySession } = require("../../middleware/verifySession");
const { verifyAdmin } = require("../../middleware/verifyAdmin");
const moment = require("moment");

const generateRef = (length) => {
  var a =
    "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890".split("");
  var b = [];
  for (var i = 0; i < length; i++) {
    var j = (Math.random() * (a.length - 1)).toFixed(0);
    b[i] = a[j];
  }
  return b.join("")?.toLowerCase();
};

// admin get any user card details
Router.get(
  "/get/card/:userID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findUser = await DB.USER.findById(req.params.userID);
      if (!findUser) throw new Error("Can't find user with the required ID");

      const carrd = await DB.CARD.findOne({ uid: findUser._id });

      res.status(201).json({
        data: carrd,
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin approve new card by admin
Router.post(
  "/post/approve/card/:userID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findUser = await DB.USER.findById(req.params.userID);
      if (!findUser) throw new Error("Can't find user with the required ID");

      let { error, value } = ACCEPTED_CREATE_CARD_SCHEMA.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      await DB.CARD.findOneAndDelete({ uid: findUser._id });

      await DB.CARD.create({
        uid: findUser._id,
        ...value,
        status: value.status,
      });

      await DB.NOTIFICATION.create({
      uid: findUser._id,
        type: "alert",
        body: {
          title: "Your Card Has Been Approved!",
          // READING AND REPLACING WHERE {{NAME FOR THE REAL NAME}} ALL THE FILE SYSTEM
          text: "🎉 Congratulations! Your card has been approved and is ready to use. Start exploring your benefits now!",
        },
      });

      res.status(201).json({
        data: "Card approved successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// edit card details by admin
Router.put(
  "/edit/card/:cardID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findCard = await DB.CARD.findById(req.params.cardID);
      if (!findCard) throw new Error("Can't find user with the required ID");

      let { error, value } = ACCEPTED_EDIT_CARD_SCHEMA.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      await DB.CARD.findByIdAndUpdate(
        findCard._id,
        { ...value },
        { new: true }
      );

      res.status(201).json({
        data: "data updated successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// deleting card of any user details by admin
Router.delete(
  "/delete/card/:cardID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findCard = await DB.CARD.findById(req.params.cardID);
      if (!findCard) throw new Error("Can't find user with the required ID");

      // check if the card has amount and use the amount and add to the user usd balance
      if (findCard.amount > 0) {
        await DB.USER.findByIdAndUpdate(
          findCard.uid,
          {
            $inc: {
              "account.balance.usd": findCard.amount,
            },
          },
          { new: true }
        );
      }

      await DB.CARD.findByIdAndDelete(findCard._id);

      res.status(201).json({
        data: "data deleted successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin post card history
Router.post(
  "/post/card/transfer/:cardID",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findCard = await DB.CARD.findById(req.params.cardID);
      if (!findCard) throw new Error("Can't find user with the required ID");

      let { error, value } = ACCEPTED_CARD_TRANSFER.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      const card_history = [...findCard.history];
      const ref = generateRef(17);

      const reValued = {
        ...value,
        ref,
        index: card_history.length,
        date: moment().format("lll"),
      };

      card_history.push(reValued);

      await DB.CARD.findByIdAndUpdate(
        findCard._id,
        { history: card_history },
        { new: true }
      );

      res.status(201).json({
        data: "Data added successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin delete card history
Router.delete(
  "/post/card/transfer/:cardID/:ref",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findCard = await DB.CARD.findById(req.params.cardID);
      if (!findCard) throw new Error("Can't find user with the required ID");

      const card_history = [...findCard.history];

      // find ref
      const findIndex = card_history.findIndex((v) => v.ref == req.params.ref);
      if (findIndex == -1) throw new Error("Can't find reference ID");

      // Remove the object with the matching ref
      const filteredArray = card_history.filter(
        (item) => item.ref !== req.params.ref
      );

      await DB.CARD.findByIdAndUpdate(
        findCard._id,
        { history: filteredArray },
        { new: true }
      );

      res.status(201).json({
        data: "Data deleted successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin editing each card transaction history
Router.put(
  "/put/card/transfer/:cardID/:ref",
  verifySession,
  verifyAdmin,
  async (req, res) => {
    try {
      const findCard = await DB.CARD.findById(req.params.cardID);
      if (!findCard) throw new Error("Can't find user with the required ID");

      const card_history = [...findCard.history];

      // find ref
      const findIndex = card_history.findIndex((v) => v.ref == req.params.ref);
      if (findIndex == -1) throw new Error("Can't find reference ID");

      let { error, value } = ACCEPTED_CARD_TRANSFER.validate(req.body);
      if (error) throw new Error(error.details[0].message);

      // Remove the object with the matching ref
      let filteredArray = card_history.filter(
        (item) => item.ref == req.params.ref
      );

      card_history.splice(findIndex, 1, {
        ...value,
        date: filteredArray?.date,
        ref: filteredArray?.ref,
        index: filteredArray?.index,
      });

      await DB.CARD.findByIdAndUpdate(
        findCard._id,
        { history: card_history },
        { new: true }
      );

      res.status(201).json({
        data: "Data updated successfully",
      });
    } catch (error) {
      res.status(500).json({
        msg: error.message || "unable to complete request",
      });
    }
  }
);

// admin getting all cards
Router.get("/get/all/cards", verifySession, verifyAdmin, async (req, res) => {
  try {
    const findCard = await DB.CARD.find()
      .populate("uid", "name email phone account")
      .sort({ createdAt: -1 });

    res.status(201).json({
      data: findCard,
    });
  } catch (error) {
    res.status(500).json({
      msg: error.message || "unable to complete request",
    });
  }
});

module.exports = Router;
